Rem ImageCatalog.vbs
Rem An InDesign VBScript
Rem
Rem Creates an image catalog from the graphic files in a selected folder.
Rem Each file can be labeled with the file name, and the labels are placed on
Rem a separate layer and formatted using a paragraph style ("label") you can
Rem modify to change the appearance of the labels.
Rem
Rem For more on InDesign/InCopy scripting see the documentation included in the Scripting SDK 
Rem available at http://www.adobe.com/devnet/indesign/sdk.html
Rem Or visit the InDesign Scripting User to User forum at http://www.adobeforums.com .
Rem
Rem The myExtensions array contains the extensions of the graphic file types you want
Rem to include in the catalog. You can remove extensions from or add extensions to this list.
myExtensions = Array(".jpg", ".jpeg", ".eps", ".ps", ".pdf", ".tif", ".tiff", ".gif", ".psd", ".ai")
main
Function main()
	ReDim myFileList(0)
	Set myInDesign = CreateObject("InDesign.Application.2024")
	Rem Set the user interaction level to allow the display of dialog boxes and alerts.
	myInDesign.ScriptPreferences.UserInteractionLevel = idUserInteractionLevels.idInteractWithAll	
	Set myFileSystemObject = CreateObject("Scripting.FileSystemObject")
	Rem Creating a folder browser in VBScript can be a problem (relying on either Windows API calls
	Rem or use of ActiveX controls which may not be present on a given system). Instead, we'll use
	Rem InDesign's built-in JavaScript to display a file browser. DoScript can return a value;
	Rem in this example, it's the name of the folder.
	myJavaScript = "myFolder = Folder.selectDialog(""Choose a Folder""); myFolderName = myFolder.fsName;"
	Rem The following line shows how to return a value from a script called with the DoScript method.
	myFolderName = myInDesign.doScript(myJavaScript, idScriptLanguage.idJavascript)
	If myFileSystemObject.FolderExists(myFolderName) Then
		Set myFolder = myFileSystemObject.GetFolder(myFolderName)
		For Each File In myFolder.Files
			myFilePath = File.Path
			If myCheckFileType(myFilePath, myExtensions) = True Then
				Rem Add the file to the list of files to be placed.
				If Not (IsEmpty(myFileList(0))) Then
					ReDim Preserve myFileList(UBound(myFileList) + 1)
				End If
				myFileList(UBound(myFileList)) = myFilePath
			End If
		Next
		If Not (IsEmpty(myFileList(0))) Then
			myDisplayDialog myInDesign, myFileList, myFolderName
		Else
			MsgBox "No files found to place in the specified folder."
		End If
	End If
End Function
Function myDisplayDialog(myInDesign, myFileList, myFolderName)
    myLabelWidth = 130
    Set myDialog = myInDesign.dialogs.Add
    myDialog.Name = "Image Catalog"
    myStyleNames = myGetStyleNames(myInDesign)
    myLabelTypes = Array("File Name", "File Path", "XMP Description", "XMP Author")
    myLayerNames = Array("Layer 1", "Labels")
    With myDialog.DialogColumns.Add
        With .DialogRows.Add
            With .DialogColumns.Add
                With .StaticTexts.Add
                    .staticLabel = "Information"
                End With
            End With
        End With
        With .BorderPanels.Add
            With .DialogColumns.Add
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Source Folder:"
                        .MinWidth = myLabelWidth
                    End With
                    With .StaticTexts.Add
                        .staticLabel = myFolderName
                    End With
                End With
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Number of Images:"
                        .MinWidth = myLabelWidth
                    End With
                    With .StaticTexts.Add
                        .staticLabel = CStr(UBound(myFileList)+1)
                    End With
                End With
            End With
        End With
        With .DialogRows.Add
            With .StaticTexts.Add
                .staticLabel = "Options"
            End With
        End With
        With .BorderPanels.Add
            With .DialogColumns.Add
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Number of Rows:"
                        .MinWidth = myLabelWidth
                    End With
                    Set myNumberOfRowsField = .integerEditboxes.Add
                    myNumberOfRowsField.editValue = 3
                End With
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Number of Columns:"
                        .MinWidth = myLabelWidth
                    End With
                    Set myNumberOfColumnsField = .integerEditboxes.Add
                    myNumberOfColumnsField.editValue = 3
                End With
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Horizontal Offset:"
                        .MinWidth = myLabelWidth
                    End With
                    Set myHorizontalOffsetField = .measurementEditboxes.Add
                    myHorizontalOffsetField.editValue = 12
                    myHorizontalOffsetField.EditUnits = idMeasurementUnits.idPoints
                End With
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Vertical Offset:"
                        .MinWidth = myLabelWidth
                    End With
                    Set myVerticalOffsetField = .measurementEditboxes.Add
                    myVerticalOffsetField.editValue = 12
                    myVerticalOffsetField.EditUnits = idMeasurementUnits.idPoints
                End With
                With .DialogRows.Add
                    With .DialogColumns.Add
                        With .StaticTexts.Add
                            .staticLabel = "Fitting:"
                            .MinWidth = myLabelWidth
                        End With
                    End With
                    With .DialogColumns.Add
                        Set myFitProportionalCheckbox = .checkboxControls.Add
                        myFitProportionalCheckbox.staticLabel = "Proportional"
                        myFitProportionalCheckbox.checkedState = True
                        Set myFitCenterContentCheckbox = .checkboxControls.Add
                        myFitCenterContentCheckbox.staticLabel = "Center Content"
                        myFitCenterContentCheckbox.checkedState = True
                        Set myFitFrameToContentCheckbox = .checkboxControls.Add
                        myFitFrameToContentCheckbox.staticLabel = "Frame to Content"
                        myFitFrameToContentCheckbox.checkedState = True
                    End With
                End With
                With .DialogRows.Add
                    Set myRemoveEmptyFramesCheckbox = .checkboxControls.Add
                    myRemoveEmptyFramesCheckbox.staticLabel = "Remove Empty Frames"
                    myRemoveEmptyFramesCheckbox.checkedState = True
                End With
			End With
		End With
		Rem "Spacer" line to separate Labels group from the border.
		With .StaticTexts.Add
			.StaticLabel = ""
		End With
        Set myLabelsGroup = .EnablingGroups.Add
        With myLabelsGroup
        	.StaticLabel = "Labels"
        	.CheckedState = True
        	With .DialogColumns.Add
        		Rem Label type dropdown and associated static text.
        		With .DialogRows.Add
        			With .DialogColumns.Add
        				With .StaticTexts.Add
        					.StaticLabel = "Label Type:"
        					.MinWidth = myLabelWidth
        				End With
        			End With
        			With .DialogColumns.Add
        				Set myLabelTypeDropdown = .Dropdowns.Add
        				With myLabelTypeDropdown
        					.StringList = myLabelTypes
        					.SelectedIndex = 0
        				End With
        			End With
        		End With
        		Rem Label text frame height and associated static text.
        		With .DialogRows.Add
        			With .DialogColumns.Add
        				With .StaticTexts.Add
        					.StaticLabel = "Label Height:"
        					.MinWidth = myLabelWidth
        				End With
        			End With
        			With .DialogColumns.Add
        				Set myLabelHeightField = .MeasurementEditboxes.Add
        				With myLabelHeightField
        					.EditValue = 24
        					.EditUnits = idMeasurementUnits.idPoints
        				End With
        			End With
        		End With
        		Rem Label text frame offset and associated static text.
        		With .DialogRows.Add
        			With .DialogColumns.Add
        				With .StaticTexts.Add
        					.StaticLabel = "Label Offset:"
        					.MinWidth = myLabelWidth
        				End With
        			End With
        			With .DialogColumns.Add
        				Set myLabelOffsetField = .MeasurementEditboxes.Add
        				With myLabelOffsetField
        					.EditValue = 0
        					.EditUnits = idMeasurementUnits.idPoints
        				End With
        			End With
        		End With
        		Rem Label paragraph style and associated static text.
        		With .DialogRows.Add
        			With .DialogColumns.Add
        				With .StaticTexts.Add
        					.StaticLabel = "Label Style:"
        					.MinWidth = myLabelWidth
        				End With
        			End With
        			With .DialogColumns.Add
        				Set myLabelStyleDropdown = .Dropdowns.Add
        				With myLabelStyleDropdown
        					.StringList = myStyleNames
        					.SelectedIndex = 0
        				End With
        			End With
        		End With
        		Rem Label layer and associated static text.
        		With .DialogRows.Add
        			With .DialogColumns.Add
        				With .StaticTexts.Add
        					.StaticLabel = "Label Layer:"
        					.MinWidth = myLabelWidth
        				End With
        			End With
        			With .DialogColumns.Add
        				Set myLabelLayerDropdown = .Dropdowns.Add
        				With myLabelLayerDropdown
        					.StringList = myLayerNames
        					.SelectedIndex = 0
        				End With
        			End With
        		End With
        	End With
		End With
    End With
    myResult = myDialog.Show
    If myResult = True Then
        myNumberOfRows = myNumberOfRowsField.editValue
        myNumberOfColumns = myNumberOfColumnsField.editValue
        myRemoveEmptyFrames = myRemoveEmptyFramesCheckbox.checkedState
        myFitProportional = myFitProportionalCheckbox.checkedState
        myFitCenterContent = myFitCenterContentCheckbox.checkedState
        myFitFrameToContent = myFitFrameToContentCheckbox.checkedState
        myHorizontalOffset = myHorizontalOffsetField.editValue
        myVerticalOffset = myVerticalOffsetField.editValue
        Rem Label options.
        myMakeLabels = myLabelsGroup.CheckedState
        myLabelHeight = myLabelHeightField.EditValue
        myLabelOffset = myLabelOffsetField.EditValue
        myLabelType = myLabelTypes(myLabelTypeDropdown.SelectedIndex)
        myLabelStyleName = myStyleNames(myLabelStyleDropdown.SelectedIndex)
        myLabelLayerName = myLayerNames(myLabelLayerDropdown.SelectedIndex)
        myDialog.destroy
        myMakeImageCatalog myInDesign, myFileList, myNumberOfRows, myNumberOfColumns, myRemoveEmptyFrames, myFitProportional, myFitCenterContent, myFitFrameToContent, myHorizontalOffset, myVerticalOffset, myMakeLabels, myLabelHeight, myLabelOffset, myLabelType, myLabelStyleName, myLabelLayerName
        MsgBox "Done!"
    Else
        myDialog.destroy
    End If
End Function
Function myMakeImageCatalog(myInDesign, myFileList, myNumberOfRows, myNumberOfColumns, myRemoveEmptyFrames, myFitProportional, myFitCenterContent, myFitFrameToContent, myHorizontalOffset, myVerticalOffset, myMakeLabels, myLabelHeight, myLabelOffset, myLabelType, myLabelStyleName, myLabelLayerName)
    Set myFileSystemObject = CreateObject("Scripting.FileSystemObject")
    Set myDocument = myInDesign.documents.Add
    myDocument.viewPreferences.horizontalMeasurementUnits = idMeasurementUnits.idPoints
    myDocument.viewPreferences.verticalMeasurementUnits = idMeasurementUnits.idPoints
    Set myDocumentPreferences = myDocument.documentPreferences
    myFramesPerPage = myNumberOfRows * myNumberOfColumns
    myNumberOfFrames = UBound(myFileList)+1
    myNumberOfPages = Round(myNumberOfFrames / myFramesPerPage)
    If (myNumberOfPages * myFramesPerPage) < myNumberOfFrames Then
        myNumberOfPages = myNumberOfPages + 1
    End If
    myDocumentPreferences.pagesPerDocument = CInt(myNumberOfPages)
    myDocumentPreferences.facingPages = False
    Set myPage = myDocument.Pages.Item(1)
    Set myMarginPreferences = myPage.marginPreferences
    myLeftMargin = myMarginPreferences.Left
    myTopMargin = myMarginPreferences.Top
    myRightMargin = myMarginPreferences.Right
    myBottomMargin = myMarginPreferences.bottom
    myLiveWidth = (myDocumentPreferences.pageWidth - (myLeftMargin + myRightMargin)) + myHorizontalOffset
    myLiveHeight = myDocumentPreferences.pageHeight - (myTopMargin + myBottomMargin)
    myColumnWidth = myLiveWidth / myNumberOfColumns
    myFrameWidth = myColumnWidth - myHorizontalOffset
    myRowHeight = myLiveHeight / myNumberOfRows
    myFrameHeight = myRowHeight - myVerticalOffset
    Set myPages = myDocument.Pages
    If myMakeLabels = True Then
        Rem Create the label paragraph style if it does not already exist.
        Err.Clear
        On Error Resume Next
        Set myLabelStyle = myDocument.paragraphStyles.Item(myLabelStyleName)
        If Err.Number <> 0 Then
            Set myLabelStyle = myDocument.paragraphStyles.Add
            myLabelStyle.Name = myLabelStyleName
            Err.Clear
        End If
        On Error GoTo 0
        Rem Create the label layer if it does not already exist.
        Err.Clear
        On Error Resume Next
        Set myLabelLayer = myDocument.layers.Item(myLabelLayerName)
        If Err.Number <> 0 Then
            Set myLabelLayer = myDocument.layers.Add
            myLabelLayer.Name = myLabelLayerName
            Err.Clear
        End If
        On Error GoTo 0
    End If
    Rem Construct the frames in reverse order. Don't laugh--this will
    Rem save us time later (when we place the graphics).
    For myCounter = myDocument.Pages.Count To 1 Step -1
        Set myPage = myPages.Item(CInt(myCounter))
        For myRowCounter = myNumberOfRows To 1 Step -1
            myY1 = myTopMargin + (myRowHeight * (myRowCounter - 1))
            myY2 = myY1 + myFrameHeight
            For myColumnCounter = myNumberOfColumns To 1 Step -1
                myX1 = myLeftMargin + (myColumnWidth * (myColumnCounter - 1))
                myX2 = myX1 + myFrameWidth
                Set myRectangle = myPage.Rectangles.Add
                myRectangle.GeometricBounds = Array(myY1, myX1, myY2, myX2)
                myRectangle.strokeWeight = 0
                myRectangle.strokeColor = myDocument.Swatches.Item("None")
            Next
        Next
    Next
    Rem Because we constructed the frames in reverse order, rectangle 1
    Rem is the first rectangle on page 1, so we can simply iterate through
    Rem the rectangles, placing a file in each one in turn. myFiles = myFolder.Files;
    For myCounter = 1 To myNumberOfFrames
        myFile = myFileList(myCounter-1)
        Set myRectangle = myDocument.Rectangles.Item(myCounter)
        myRectangle.place myFile
        myRectangle.Label
        Rem Apply fitting options as specified.
        If myFitProportional Then
            myRectangle.fit idFitOptions.idProportionally
        End If
        If myFitCenterContent Then
            myRectangle.fit idFitOptions.idCenterContent
        End If
        If myFitFrameToContent Then
            myRectangle.fit idFitOptions.idFrameToContent
        End If
        Rem Add the label, if necessary.
        If myMakeLabels = True Then
        	Set myRegExp = New RegExp
			myRegExp.Pattern = "^\s*$"
			myRegExp.IgnoreCase = True
			myX1 = myRectangle.GeometricBounds(1)
			myY1 = myRectangle.GeometricBounds(2) + myLabelOffset
			myX2 = myRectangle.GeometricBounds(3)
			myY2 = myY1 + myLabelHeight
			Select Case myLabelType
				Case "File Name":
					myString = myFileSystemObject.GetFileName(myFile)
				Case "File Path":
					myString = myFileSystemObject.GetAbsolutePathName(myFile)
				Case "XMP Author":
					On Error Resume Next
						myString = myLink.LinkXmp.Author
						If myRegExp.Replace(myString, "") = "" Then
							Error
						End If
					If Err.Number <> 0 Then
						myString = "No author available."
						Err.Clear
					End If
					On Error Goto 0
				Case "XMP Description":
					On Error Resume Next
						myString = myLink.LinkXmp.Description
						If myRegExp.Replace(myString, "") = "" Then
							Error
						End If
					If Err.Number <> 0 Then
						myString = "No description available."
						Err.Clear
					End If
					On Error Goto 0
			End Select
			Set myTextFrame = myRectangle.Parent.textFrames.Add
			myTextFrame.ItemLayer = myLabelLayer
			myTextFrame.GeometricBounds = Array(myY1, myX1, myY2, myX2)
			myTextFrame.Contents = myString
			myTextFrame.textFramePreferences.firstBaselineOffset = idFirstBaseline.idLeadingOffset
			myTextFrame.ParentStory.Texts.Item(1).appliedParagraphStyle = myLabelStyle
        End If
    Next
    If myRemoveEmptyFrames = True Then
		For myCounter = myDocument.Rectangles.Count To 1 Step -1
			If myDocument.Rectangles.Item(myCounter).contentType = idContentType.idUnassigned Then
				myDocument.Rectangles.Item(myCounter).Delete
			Else
				Rem As soon as you encounter a rectangle with content, exit the loop.
				Exit For
			End If
		Next
    End If
End Function
Function myCheckFileType(myFilePath, myExtensions)
	myFilePath = LCase(myFilePath)
    For Each myExtension In myExtensions
        If InStr(myFilePath, myExtension) <> 0 Then
            myCheckFileType = True
            Exit Function
        End If
    Next
    myCheckFileType = False
End Function
Rem Return an array of paragraph style names.
Function myGetStyleNames(myInDesign)
	myAddLabelStyle = True
	ReDim myStyleNames(0)
	For myCounter = 1 To myInDesign.ParagraphStyles.Count
		If Not (IsEmpty(myStyleNames(0))) Then
	    	ReDim Preserve myStyleNames(UBound(myStyleNames) + 1)
	    End If
		myStyleNames(UBound(myStyleNames)) = myInDesign.ParagraphStyles.Item(myCounter).Name
		If myInDesign.ParagraphStyles.Item(myCounter).Name = "Labels" Then
			myAddLabelStyle = False
		End If
	Next
	If myAddLabelStyle = True Then
		ReDim Preserve myStyleNames(UBound(myStyleNames)+1)
		myStyleNames(UBound(myStyleNames)) = "Labels"		
	End If
	myGetStyleNames = myStyleNames
End Function
'' SIG '' Begin signature block
'' SIG '' MIIpyAYJKoZIhvcNAQcCoIIpuTCCKbUCAQExDzANBglg
'' SIG '' hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
'' SIG '' BgEEAYI3AgEeMCQCAQEEEE7wKRaZJ7VNj+Ws4Q8X66sC
'' SIG '' AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
'' SIG '' pr8D+vMk2biN+WQ5bdOStlU+XwgpCqugZnKTBxOJUs2g
'' SIG '' gg6WMIIGsDCCBJigAwIBAgIQCK1AsmDSnEyfXs2pvZOu
'' SIG '' 2TANBgkqhkiG9w0BAQwFADBiMQswCQYDVQQGEwJVUzEV
'' SIG '' MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
'' SIG '' d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2Vy
'' SIG '' dCBUcnVzdGVkIFJvb3QgRzQwHhcNMjEwNDI5MDAwMDAw
'' SIG '' WhcNMzYwNDI4MjM1OTU5WjBpMQswCQYDVQQGEwJVUzEX
'' SIG '' MBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMT
'' SIG '' OERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5n
'' SIG '' IFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMIICIjANBgkq
'' SIG '' hkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA1bQvQtAorXi3
'' SIG '' XdU5WRuxiEL1M4zrPYGXcMW7xIUmMJ+kjmjYXPXrNCQH
'' SIG '' 4UtP03hD9BfXHtr50tVnGlJPDqFX/IiZwZHMgQM+TXAk
'' SIG '' ZLON4gh9NH1MgFcSa0OamfLFOx/y78tHWhOmTLMBICXz
'' SIG '' ENOLsvsI8IrgnQnAZaf6mIBJNYc9URnokCF4RS6hnyzh
'' SIG '' GMIazMXuk0lwQjKP+8bqHPNlaJGiTUyCEUhSaN4QvRRX
'' SIG '' XegYE2XFf7JPhSxIpFaENdb5LpyqABXRN/4aBpTCfMjq
'' SIG '' GzLmysL0p6MDDnSlrzm2q2AS4+jWufcx4dyt5Big2MEj
'' SIG '' R0ezoQ9uo6ttmAaDG7dqZy3SvUQakhCBj7A7CdfHmzJa
'' SIG '' wv9qYFSLScGT7eG0XOBv6yb5jNWy+TgQ5urOkfW+0/tv
'' SIG '' k2E0XLyTRSiDNipmKF+wc86LJiUGsoPUXPYVGUztYuBe
'' SIG '' M/Lo6OwKp7ADK5GyNnm+960IHnWmZcy740hQ83eRGv7b
'' SIG '' UKJGyGFYmPV8AhY8gyitOYbs1LcNU9D4R+Z1MI3sMJN2
'' SIG '' FKZbS110YU0/EpF23r9Yy3IQKUHw1cVtJnZoEUETWJrc
'' SIG '' JisB9IlNWdt4z4FKPkBHX8mBUHOFECMhWWCKZFTBzCEa
'' SIG '' 6DgZfGYczXg4RTCZT/9jT0y7qg0IU0F8WD1Hs/q27Iwy
'' SIG '' CQLMbDwMVhECAwEAAaOCAVkwggFVMBIGA1UdEwEB/wQI
'' SIG '' MAYBAf8CAQAwHQYDVR0OBBYEFGg34Ou2O/hfEYb7/mF7
'' SIG '' CIhl9E5CMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/n
'' SIG '' upiuHA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAK
'' SIG '' BggrBgEFBQcDAzB3BggrBgEFBQcBAQRrMGkwJAYIKwYB
'' SIG '' BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBB
'' SIG '' BggrBgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQw
'' SIG '' QwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGln
'' SIG '' aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5j
'' SIG '' cmwwHAYDVR0gBBUwEzAHBgVngQwBAzAIBgZngQwBBAEw
'' SIG '' DQYJKoZIhvcNAQEMBQADggIBADojRD2NCHbuj7w6mdNW
'' SIG '' 4AIapfhINPMstuZ0ZveUcrEAyq9sMCcTEp6QRJ9L/Z6j
'' SIG '' fCbVN7w6XUhtldU/SfQnuxaBRVD9nL22heB2fjdxyyL3
'' SIG '' WqqQz/WTauPrINHVUHmImoqKwba9oUgYftzYgBoRGRjN
'' SIG '' YZmBVvbJ43bnxOQbX0P4PpT/djk9ntSZz0rdKOtfJqGV
'' SIG '' WEjVGv7XJz/9kNF2ht0csGBc8w2o7uCJob054ThO2m67
'' SIG '' Np375SFTWsPK6Wrxoj7bQ7gzyE84FJKZ9d3OVG3ZXQIU
'' SIG '' H0AzfAPilbLCIXVzUstG2MQ0HKKlS43Nb3Y3LIU/Gs4m
'' SIG '' 6Ri+kAewQ3+ViCCCcPDMyu/9KTVcH4k4Vfc3iosJocsL
'' SIG '' 6TEa/y4ZXDlx4b6cpwoG1iZnt5LmTl/eeqxJzy6kdJKt
'' SIG '' 2zyknIYf48FWGysj/4+16oh7cGvmoLr9Oj9FpsToFpFS
'' SIG '' i0HASIRLlk2rREDjjfAVKM7t8RhWByovEMQMCGQ8M4+u
'' SIG '' KIw8y4+ICw2/O/TOHnuO77Xry7fwdxPm5yg/rBKupS8i
'' SIG '' bEH5glwVZsxsDsrFhsP2JjMMB0ug0wcCampAMEhLNKhR
'' SIG '' ILutG4UI4lkNbcoFUCvqShyepf2gpx8GdOfy1lKQ/a+F
'' SIG '' SCH5Vzu0nAPthkX0tGFuv2jiJmCG6sivqf6UHedjGzqG
'' SIG '' VnhOMIIH3jCCBcagAwIBAgIQD99ExMbwCe2qSx8hp1o/
'' SIG '' EDANBgkqhkiG9w0BAQsFADBpMQswCQYDVQQGEwJVUzEX
'' SIG '' MBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMT
'' SIG '' OERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5n
'' SIG '' IFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMB4XDTIzMTEw
'' SIG '' MzAwMDAwMFoXDTI1MTEwNDIzNTk1OVowgeYxEzARBgsr
'' SIG '' BgEEAYI3PAIBAxMCVVMxGTAXBgsrBgEEAYI3PAIBAhMI
'' SIG '' RGVsYXdhcmUxHTAbBgNVBA8MFFByaXZhdGUgT3JnYW5p
'' SIG '' emF0aW9uMRAwDgYDVQQFEwcyNzQ4MTI5MQswCQYDVQQG
'' SIG '' EwJVUzELMAkGA1UECBMCY2ExETAPBgNVBAcTCFNhbiBK
'' SIG '' b3NlMRMwEQYDVQQKEwpBZG9iZSBJbmMuMSwwKgYDVQQL
'' SIG '' EyNJbGx1c3RyYXRvciwgSW5EZXNpZ24sIEluQ29weSwg
'' SIG '' TXVzZTETMBEGA1UEAxMKQWRvYmUgSW5jLjCCAiIwDQYJ
'' SIG '' KoZIhvcNAQEBBQADggIPADCCAgoCggIBAPCR5c/cgyg4
'' SIG '' MpJD46/+BWENPhE1RjN77VE2BWlIUdiO2nlWelKthEq9
'' SIG '' aK+1ofKIlDVc9mvxksuXBLESi7HBt3PCVBntNJkjBJd0
'' SIG '' EXeprmDK8HZrz8y/KSwEewbTMjneMRIRcJV2GMoAyUzK
'' SIG '' EyFJF7m6ByeoNBDDXMD4AQF5C2l3wfvRbREw+G+cyeya
'' SIG '' 564AHpFJ9sfqi9jAycnEZISxEf6N72GgGQr04cZt3JTb
'' SIG '' TktZGt6uWdZnseXDLGRHzFu4P4EqmVKvfMDliaKyWBzV
'' SIG '' lY+gmtE1vSxIK79WTySFrFY5j1vtbKyHybk6tv4TOmLt
'' SIG '' Lw4OP/ngVDiw/ZP+b7saGxPOEfuh5LOAr+llt+6V5NhT
'' SIG '' dkYPR6cbokawwX2DbGg7IeVuUPdf9bx/LznFV/8kPxCB
'' SIG '' J6qg0dqvCmQt/XbBXPJUvDoVJ9z+9hDlrAoLESqpHK8U
'' SIG '' 73UFGohBMJ/7itIyKngem7Ms98z6x0i63FJA8C06g+XE
'' SIG '' GJUmR28+c6ba+m/sc6xHI4zdL5a84UeRS9+u8UL8N4av
'' SIG '' fpSxHYTTxVjk1ZlwQ8dLARwEVWvTu4lRcEck3RPogx7o
'' SIG '' o8wUpSw9++m+Oi+K/z7RP/R67uCEtrkdM5N/xUoQR/t6
'' SIG '' 3P2ia9g8+LuYJ2BNBd1+qMcSoSlDfIFLFMeeqESbMv+j
'' SIG '' UsRQJn2lZvufAgMBAAGjggICMIIB/jAfBgNVHSMEGDAW
'' SIG '' gBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4EFgQU
'' SIG '' RPnrJ9yT26rMrd/Q4PfQSlgeaRcwPQYDVR0gBDYwNDAy
'' SIG '' BgVngQwBAzApMCcGCCsGAQUFBwIBFhtodHRwOi8vd3d3
'' SIG '' LmRpZ2ljZXJ0LmNvbS9DUFMwDgYDVR0PAQH/BAQDAgeA
'' SIG '' MBMGA1UdJQQMMAoGCCsGAQUFBwMDMIG1BgNVHR8Ega0w
'' SIG '' gaowU6BRoE+GTWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNv
'' SIG '' bS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWduaW5nUlNB
'' SIG '' NDA5NlNIQTM4NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRw
'' SIG '' Oi8vY3JsNC5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVz
'' SIG '' dGVkRzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIx
'' SIG '' Q0ExLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQwJAYIKwYB
'' SIG '' BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBc
'' SIG '' BggrBgEFBQcwAoZQaHR0cDovL2NhY2VydHMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25p
'' SIG '' bmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcnQwCQYDVR0T
'' SIG '' BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAPIZ9C4IuKg85
'' SIG '' 0H8M66OLMVllIpANmEjijIG/D65qP6jfma+2WFps2bb+
'' SIG '' vKKIEtG9TFtCAlSrA1OV6Ru78paq0gqw/q3QAN0OleFR
'' SIG '' F1NvT/TLO0ulhjXTETB1BwSKZaJH001DH0+60TmbGfVh
'' SIG '' mwLrEj2Zfl4COFwivZ/2tRoknZl5tmiu4hpjDbOapWd9
'' SIG '' 5cqliSoMeu2i588557eikdiDk6bXBa5MxHjyXvbvaIWN
'' SIG '' Zjsymn0L0sYkF3BToDN0h0XiTTZjZPHdSV2GjX6C/+Nj
'' SIG '' HSZGRtBaIxdAL1mKEWJV9qkEKELPb098rO+zGdn1E5IH
'' SIG '' pS1zSRIYngwZEjZfyoCkg1F+EW4QPs2+ff2CsW5i3IEd
'' SIG '' 7vL19w9oyce3S4wejvnlTKt+EQWn3gSPxrbkp3BxdWRN
'' SIG '' 25MKo5bS5bJ21BiwqcIJHcTxWNgaQPh4sZboi9IimCSn
'' SIG '' UenPQhqdeOuz0NnmepMEYd7L/kqnAeUZ8bW1UbITAWQ/
'' SIG '' UT3phjCfRQNpEh4OBoUcH8hOOHZ0Lh86mxv9lDP3i7c/
'' SIG '' bQzZnrJr2dS34JXolwREbTX9Yu5JB5X3YP539/M1Ms5O
'' SIG '' FZou/MHQBrWMsVBsEXDTWqwbMTYw/zvIiHXcGUnpzInw
'' SIG '' 1jQXlLrmwVHW4ug+A4LFO7bWC4mnZoDNWp0xnlGmbP6I
'' SIG '' ccJmW5C7RJMxghqKMIIahgIBATB9MGkxCzAJBgNVBAYT
'' SIG '' AlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFBMD8G
'' SIG '' A1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNp
'' SIG '' Z25pbmcgUlNBNDA5NiBTSEEzODQgMjAyMSBDQTECEA/f
'' SIG '' RMTG8AntqksfIadaPxAwDQYJYIZIAWUDBAIBBQCggZww
'' SIG '' GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYB
'' SIG '' BAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcN
'' SIG '' AQkEMSIEIACL46N7AKftOS6NShCGO2SzWRgfy+rtTCS5
'' SIG '' GmKNwmn6MDAGCisGAQQBgjcCAQwxIjAgoB6AHABBAGQA
'' SIG '' bwBiAGUAIABJAG4ARABlAHMAaQBnAG4wDQYJKoZIhvcN
'' SIG '' AQEBBQAEggIATPPUFAzNGr+MM+CEVjAs3pud/IaVCvQI
'' SIG '' 3uEvnY85QafWkSIc5GcE/HkUVmriAyNgePF9obrS2Y8K
'' SIG '' HoPf1S0fhl7eBiR9QknDx64FmFXcz5vGhnnHf5KRAf/r
'' SIG '' JYrQvhbQ9omwCq+0xCk7jolM/QW7l2htBrGfQVH4jLtE
'' SIG '' dy/lrCzRV/ueEAPBc50xl+3Fn+7apaMQvN2LrxoptqxL
'' SIG '' w+RiMgbud3e0PXzI/v2qN/WADt3xT1yvX18XRJvcXS59
'' SIG '' G7EnfoGRnX7I/XTwLo2/O0gYH8K4eYyR5/mODVa7SsIn
'' SIG '' gfBxt4J/LtLVB/UQFvtUshuei0e8fQNM2jDV8/veYg0F
'' SIG '' YH4lzaVXFHoFANi3hRSXFFn/bon6qZve518S739TIkuc
'' SIG '' QC5+k5c58+U54zGDRX/YCLOiN1mO07AU4ry3GarAvjj4
'' SIG '' kly+KMJm1KzeeWg8U47YTNiDgqDWhvbiVtMBljeqipar
'' SIG '' 9DAjU9YJUGZmNTJL2qIOQDLcQu9perxuL1RmhTobS3EJ
'' SIG '' ykdh4Z+RXu8iPtsQPLryIhy5dDG+3Ijs5FHh6uIutezw
'' SIG '' MyVsefXnwKyPxzn5b1lsEcvhZlE9dsX6ISdZys/Tu6wW
'' SIG '' 7SOBJSFvB8EAB9e5+bz8dfIsteIyNn2TzIq8qzb1uyU4
'' SIG '' FdozZ8tpePItBIyefrWBGycUJb3/f8EwQdmhghc/MIIX
'' SIG '' OwYKKwYBBAGCNwMDATGCFyswghcnBgkqhkiG9w0BBwKg
'' SIG '' ghcYMIIXFAIBAzEPMA0GCWCGSAFlAwQCAQUAMHcGCyqG
'' SIG '' SIb3DQEJEAEEoGgEZjBkAgEBBglghkgBhv1sBwEwMTAN
'' SIG '' BglghkgBZQMEAgEFAAQgx/Ow2rsJWEcdsUEfwNHhl89D
'' SIG '' jxjhMoAV+TXaJWuxiOwCEDCeAtTG378/SLUDpTssmqsY
'' SIG '' DzIwMjQwMTI2MjM1NDQxWqCCEwkwggbCMIIEqqADAgEC
'' SIG '' AhAFRK/zlJ0IOaa/2z9f5WEWMA0GCSqGSIb3DQEBCwUA
'' SIG '' MGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2Vy
'' SIG '' dCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
'' SIG '' ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcg
'' SIG '' Q0EwHhcNMjMwNzE0MDAwMDAwWhcNMzQxMDEzMjM1OTU5
'' SIG '' WjBIMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
'' SIG '' cnQsIEluYy4xIDAeBgNVBAMTF0RpZ2lDZXJ0IFRpbWVz
'' SIG '' dGFtcCAyMDIzMIICIjANBgkqhkiG9w0BAQEFAAOCAg8A
'' SIG '' MIICCgKCAgEAo1NFhx2DjlusPlSzI+DPn9fl0uddoQ4J
'' SIG '' 3C9Io5d6OyqcZ9xiFVjBqZMRp82qsmrdECmKHmJjadNY
'' SIG '' nDVxvzqX65RQjxwg6seaOy+WZuNp52n+W8PWKyAcwZeU
'' SIG '' tKVQgfLPywemMGjKg0La/H8JJJSkghraarrYO8pd3hkY
'' SIG '' hftF6g1hbJ3+cV7EBpo88MUueQ8bZlLjyNY+X9pD04T1
'' SIG '' 0Mf2SC1eRXWWdf7dEKEbg8G45lKVtUfXeCk5a+B4WZfj
'' SIG '' RCtK1ZXO7wgX6oJkTf8j48qG7rSkIWRw69XloNpjsy7p
'' SIG '' Be6q9iT1HbybHLK3X9/w7nZ9MZllR1WdSiQvrCuXvp/k
'' SIG '' /XtzPjLuUjT71Lvr1KAsNJvj3m5kGQc3AZEPHLVRzapM
'' SIG '' ZoOIaGK7vEEbeBlt5NkP4FhB+9ixLOFRr7StFQYU6mII
'' SIG '' E9NpHnxkTZ0P387RXoyqq1AVybPKvNfEO2hEo6U7Qv1z
'' SIG '' fe7dCv95NBB+plwKWEwAPoVpdceDZNZ1zY8SdlalJPrX
'' SIG '' xGshuugfNJgvOuprAbD3+yqG7HtSOKmYCaFxsmxxrz64
'' SIG '' b5bV4RAT/mFHCoz+8LbH1cfebCTwv0KCyqBxPZySkwS0
'' SIG '' aXAnDU+3tTbRyV8IpHCj7ArxES5k4MsiK8rxKBMhSVF+
'' SIG '' BmbTO77665E42FEHypS34lCh8zrTioPLQHsCAwEAAaOC
'' SIG '' AYswggGHMA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8E
'' SIG '' AjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMCAGA1Ud
'' SIG '' IAQZMBcwCAYGZ4EMAQQCMAsGCWCGSAGG/WwHATAfBgNV
'' SIG '' HSMEGDAWgBS6FtltTYUvcyl2mi91jGogj57IbzAdBgNV
'' SIG '' HQ4EFgQUpbbvE+fvzdBkodVWqWUxo97V40kwWgYDVR0f
'' SIG '' BFMwUTBPoE2gS4ZJaHR0cDovL2NybDMuZGlnaWNlcnQu
'' SIG '' Y29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1
'' SIG '' NlRpbWVTdGFtcGluZ0NBLmNybDCBkAYIKwYBBQUHAQEE
'' SIG '' gYMwgYAwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRp
'' SIG '' Z2ljZXJ0LmNvbTBYBggrBgEFBQcwAoZMaHR0cDovL2Nh
'' SIG '' Y2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3Rl
'' SIG '' ZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNy
'' SIG '' dDANBgkqhkiG9w0BAQsFAAOCAgEAgRrW3qCptZgXvHCN
'' SIG '' T4o8aJzYJf/LLOTN6l0ikuyMIgKpuM+AqNnn48XtJoKK
'' SIG '' cS8Y3U623mzX4WCcK+3tPUiOuGu6fF29wmE3aEl3o+uQ
'' SIG '' qhLXJ4Xzjh6S2sJAOJ9dyKAuJXglnSoFeoQpmLZXeY/b
'' SIG '' JlYrsPOnvTcM2Jh2T1a5UsK2nTipgedtQVyMadG5K8TG
'' SIG '' e8+c+njikxp2oml101DkRBK+IA2eqUTQ+OVJdwhaIcW0
'' SIG '' z5iVGlS6ubzBaRm6zxbygzc0brBBJt3eWpdPM43UjXd9
'' SIG '' dUWhpVgmagNF3tlQtVCMr1a9TMXhRsUo063nQwBw3syY
'' SIG '' nhmJA+rUkTfvTVLzyWAhxFZH7doRS4wyw4jmWOK22z75
'' SIG '' X7BC1o/jF5HRqsBV44a/rCcsQdCaM0qoNtS5cpZ+l3k4
'' SIG '' SF/Kwtw9Mt911jZnWon49qfH5U81PAC9vpwqbHkB3NpE
'' SIG '' 5jreODsHXjlY9HxzMVWggBHLFAx+rrz+pOt5Zapo1iLK
'' SIG '' O+uagjVXKBbLafIymrLS2Dq4sUaGa7oX/cR3bBVsrquv
'' SIG '' czroSUa31X/MtjjA2Owc9bahuEMs305MfR5ocMB3CtQC
'' SIG '' 4Fxguyj/OOVSWtasFyIjTvTs0xf7UGv/B3cfcZdEQcm4
'' SIG '' RtNsMnxYL2dHZeUbc7aZ+WssBkbvQR7w8F/g29mtkIBE
'' SIG '' r4AQQYowggauMIIElqADAgECAhAHNje3JFR82Ees/Shm
'' SIG '' Kl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNVBAYTAlVT
'' SIG '' MRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsT
'' SIG '' EHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lD
'' SIG '' ZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMjAzMjMwMDAw
'' SIG '' MDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVT
'' SIG '' MRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UE
'' SIG '' AxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNI
'' SIG '' QTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
'' SIG '' DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXH
'' SIG '' JQPE8pE3qZdRodbSg9GeTKJtoLDMg/la9hGhRBVCX6SI
'' SIG '' 82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0hNoR
'' SIG '' 8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU
'' SIG '' 5ygt69OxtXXnHwZljZQp09nsad/ZkIdGAHvbREGJ3Hxq
'' SIG '' V3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAshaG43IbtA
'' SIG '' rF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECn
'' SIG '' wHLFuk4fsbVYTXn+149zk6wsOeKlSNbwsDETqVcplicu
'' SIG '' 9Yemj052FVUmcJgmf6AaRyBD40NjgHt1biclkJg6OBGz
'' SIG '' 9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpUR
'' SIG '' K1h0QCirc0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD
'' SIG '' 4L/wojzKQtwYSH8UNM/STKvvmz3+DrhkKvp1KCRB7UK/
'' SIG '' BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T
'' SIG '' /jnA+bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uK
'' SIG '' IqjBJgj5FBASA31fI7tk42PgpuE+9sJ0sj8eCXbsq11G
'' SIG '' deJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
'' SIG '' M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB
'' SIG '' /wIBADAdBgNVHQ4EFgQUuhbZbU2FL3MpdpovdYxqII+e
'' SIG '' yG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4c
'' SIG '' D08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsG
'' SIG '' AQUFBwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEFBQcw
'' SIG '' AYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsG
'' SIG '' AQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5j
'' SIG '' b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNydDBDBgNV
'' SIG '' HR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2Vy
'' SIG '' dC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAg
'' SIG '' BgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEw
'' SIG '' DQYJKoZIhvcNAQELBQADggIBAH1ZjsCTtm+YqUQiAX5m
'' SIG '' 1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxp
'' SIG '' wc8dB+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGId
'' SIG '' DAiCqBa9qVbPFXONASIlzpVpP0d3+3J0FNf/q0+KLHqr
'' SIG '' hc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
'' SIG '' 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeC
'' SIG '' RK6ZJxurJB4mwbfeKuv2nrF5mYGjVoarCkXJ38SNoOeY
'' SIG '' +/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3ZpHx
'' SIG '' cpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fx
'' SIG '' ZsNBzU+2QJshIUDQtxMkzdwdeDrknq3lNHGS1yZr5Dhz
'' SIG '' q6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc4d0j/R0o
'' SIG '' 08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1Oby
'' SIG '' F5lZynDwN7+YAN8gFk8n+2BnFqFmut1VwDophrCYoCvt
'' SIG '' lUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZvAT6gt4y3wSJ
'' SIG '' 8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8
'' SIG '' mJb2VVQrH4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt
'' SIG '' 1nz8MIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAY
'' SIG '' WjANBgkqhkiG9w0BAQwFADBlMQswCQYDVQQGEwJVUzEV
'' SIG '' MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
'' SIG '' d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2Vy
'' SIG '' dCBBc3N1cmVkIElEIFJvb3QgQ0EwHhcNMjIwODAxMDAw
'' SIG '' MDAwWhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJV
'' SIG '' UzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQL
'' SIG '' ExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdp
'' SIG '' Q2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3
'' SIG '' DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAi
'' SIG '' MGkz7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf
'' SIG '' 8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4VpX6+
'' SIG '' n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7
'' SIG '' bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7Fsa
'' SIG '' vOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3T6cw2Vbu
'' SIG '' yntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jHtrHE
'' SIG '' tWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6R
'' SIG '' AXwhTNS8rhsDdV14Ztk6MUSaM0C/CNdaSaTC5qmgZ92k
'' SIG '' J7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI
'' SIG '' 8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppE
'' SIG '' GSt+wJS00mFt6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LV
'' SIG '' jHAsQWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QY
'' SIG '' uKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+N
'' SIG '' P8m800ERElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtv
'' SIG '' sauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmp
'' SIG '' sh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB
'' SIG '' /zAdBgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08w
'' SIG '' HwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNtyA8w
'' SIG '' DgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAk
'' SIG '' BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQu
'' SIG '' Y29tMEMGCCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0cy5k
'' SIG '' aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290
'' SIG '' Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9j
'' SIG '' cmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJ
'' SIG '' RFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRVHSAAMA0G
'' SIG '' CSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW
'' SIG '' 1/e/Vwe9mqyhhyzshV6pGrsi+IcaaVQi7aSId229GhT0
'' SIG '' E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO0Cre+i1Wz/n0
'' SIG '' 96wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE
'' SIG '' 1Od/6Fmo8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLih
'' SIG '' Vo7spNU96LHc/RzY9HdaXFSMb++hUD38dglohJ9vytsg
'' SIG '' jTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt
'' SIG '' 9H5xaiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cB
'' SIG '' qZ9Xql4o4rmUMYIDdjCCA3ICAQEwdzBjMQswCQYDVQQG
'' SIG '' EwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5
'' SIG '' BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5
'' SIG '' NiBTSEEyNTYgVGltZVN0YW1waW5nIENBAhAFRK/zlJ0I
'' SIG '' Oaa/2z9f5WEWMA0GCWCGSAFlAwQCAQUAoIHRMBoGCSqG
'' SIG '' SIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0B
'' SIG '' CQUxDxcNMjQwMTI2MjM1NDQxWjArBgsqhkiG9w0BCRAC
'' SIG '' DDEcMBowGDAWBBRm8CsywsLJD4JdzqqKycZPGZzPQDAv
'' SIG '' BgkqhkiG9w0BCQQxIgQgYWxsLF9z+NwcpKanlt7BkKjL
'' SIG '' 4JrEoAxRxpTi7pXCnqMwNwYLKoZIhvcNAQkQAi8xKDAm
'' SIG '' MCQwIgQg0vbkbe10IszR1EBXaEE2b4KK2lWarjMWr00a
'' SIG '' mtQMeCgwDQYJKoZIhvcNAQEBBQAEggIAmb35XFTN2BbV
'' SIG '' PCEkpuuolJQYsHI7o0uODH2Yt33sWotMTRgQOHBqmSSi
'' SIG '' sZljWT/gZ382Lle1wCvXA1KDl7iP16dMGJwy0F97rMAB
'' SIG '' RSDNwEO8xfOv7Uyo/v3YBoXt+uV+KwrFd14V3fNRCtYG
'' SIG '' vAu02GEBB6EUQb0gL6V8G8OdORWBJcDuxsvZ/wcWMeDu
'' SIG '' FF/7Ck3JTuAUyLI1z7AQUeSYm2mEXJKMLWUWnlytJguA
'' SIG '' uXDMDDOUZ9u5CoJJqj/fGSun9v6z26gzYOpC6GprKvYX
'' SIG '' Fl3iW7iTrf8Qrkt/NWRm+LqAcZc92g98sgwJIt3/Dluy
'' SIG '' 6Iy/eu7uNR1bynfywtHrNLQQKjKmUGs5/erc7gBc50Z8
'' SIG '' TrNhKWup79GtG3RFJYS8DBXqNu5a0iVrxe0hPfrJp11h
'' SIG '' WEvgqqfDC6WwYKML2zZvJv0/1je3/m3R6wU5qXiTvOrP
'' SIG '' /pBB3AVyPaQCs6CnNjvsGze5pEbjo0ynKwFQVgZwvN/P
'' SIG '' PBVOVdtA82jZ6xYcynKvnXNiNhwcLqqqaXJRmvb1oo8i
'' SIG '' Qi/L7f1l6qmttgCHHO6piCiZuVUx+2jD5/eKuEo9AqMR
'' SIG '' F3lI+KIKQtMhbJEC4OQK9VHqiLsfYwwkmo8w6bFcZ7FG
'' SIG '' LFWLxbhlRrlgRbPjbw9KVNJHg2DWVPawzV8u2cwbL9NW
'' SIG '' GgGZ2zUrykY=
'' SIG '' End signature block
